extern "C" {

	#define WIN32_LEAN_AND_MEAN 
	#include <windows.h>
	#include "Mapper_Def.h"

	char author[]	= "PrOxY";
	char name[]		= "MMC3";

	TMapperParam	* MP			= NULL;	// Contains all variables and mapper functions
	BYTE			IRQ_Counter		= 0;	// IRQ Counter
	BYTE			IRQ_Latch		= 0;	// IRQ Latch
	BYTE			IRQ_Enabled		= 0;	// IRQ Enabled?
	BYTE			Command			= 0;	// Current Command
	
	BYTE			Bank0 = 0;				// Bank0 = 89
	BYTE			Bank2 = 0;				// Bank2 = CD
	// AB and EF = always the same (only 0 and 2 can be switched)

	
	NESTENMAPPER mapper = 
	{
		4,		// Mapper Number
		0,		// Lo Version Number
		1,		// Hi Version Number
		8192,	// Min Switch Size
		name,	// Name of this mapper
		author	// My Name :)
	};


	//-----------------------------------------------------------------------------
	// Name: CMD1Write(int Bank, int Where, int What)
	// Desc: records and executs the commands
	//-----------------------------------------------------------------------------  
	__declspec(dllexport) void CMD1Write(int Bank, int Where, int What)
	{
		bool vxor = false;
		bool pswp = false;


		switch ( Where )
		{
		case 0x000:
			Command = What;
			if ( (Command & 0x40) == 0 )
			{
				MP->SetPRG_89(Bank0);
				MP->SetPRG_CD(Bank2);
			}
			else
			{
				MP->SetPRG_CD(Bank0);
				MP->SetPRG_89(Bank2);
			}
			break;

		case 0x001:
			vxor = ( (Command & 0x80) == 0 );
			pswp = ( (Command & 0x40) == 0 );

			switch ( Command & 0x07 )
			{
			case 0:
				if (vxor)
					MP->SetCHR_0000_07FF(What >> 1);
				else
					MP->SetCHR_1000_17FF(What >> 1);
				break;

			case 1:
				if (vxor)
					MP->SetCHR_0800_0FFF(What >> 1);
				else
					MP->SetCHR_1800_1FFF(What >> 1);
				break;

			case 2:
				if (vxor)
					MP->SetCHR_1000_13FF(What);
				else
					MP->SetCHR_0000_03FF(What);
				break;

			case 3:
				if (vxor)
					MP->SetCHR_1400_17FF(What);
				else
					MP->SetCHR_0400_07FF(What);
				break;

			case 4:
				if (vxor)
					MP->SetCHR_1800_1BFF(What);
				else
					MP->SetCHR_0800_0BFF(What);
				break;

			case 5:
				if (vxor)
					MP->SetCHR_1C00_1FFF(What);
				else
					MP->SetCHR_0C00_0FFF(What);
				break;

			case 6:
				if (pswp)
				{
					Bank0 = What;
					MP->SetPRG_89(Bank0);
					MP->SetPRG_CD(Bank2);
				}
				else
				{
					Bank0 = What;
					MP->SetPRG_CD(Bank0);
					MP->SetPRG_89(Bank2);
				}
				break;
			
			case 7:
				MP->SetPRG_AB(What);
				break;
			}
			break;
		}
	}


	//-----------------------------------------------------------------------------
	// Name: CMD2Write(int Bank, int Where, int What)
	// Desc: Handles mirroring
	//-----------------------------------------------------------------------------  
	__declspec(dllexport) void CMD2Write(int Bank, int Where, int What)
	{
		switch ( Where )
		{
		case 0:
			if ( (What & 0x01) != 0 )	
				MP->Mirror_H();
			else			
				MP->Mirror_V();
			break;
		}
	}


	//-----------------------------------------------------------------------------
	// Name: CMD3Write(int Bank, int Where, int What)
	// Desc: IRQ counter/latch control
	//-----------------------------------------------------------------------------  
	__declspec(dllexport) void CMD3Write(int Bank, int Where, int What)
	{
		switch ( Where )
		{
		case 0:
			IRQ_Counter = What;//+1;
			break;
		case 1:
			IRQ_Latch	= What;//+1;
			break;
		}
	}


	//-----------------------------------------------------------------------------
	// Name: CMD4Write(int Bank, int Where, int What)
	// Desc: IRQ counter/latch control, can also disable/enable IRQ
	//-----------------------------------------------------------------------------  
	__declspec(dllexport) void CMD4Write(int Bank, int Where, int What)
	{
		switch ( Where )
		{
		case 0:		
			IRQ_Enabled = 0;
			IRQ_Counter = IRQ_Latch;
			break;
		case 1:
			IRQ_Enabled = 1;
			break;
		}
	}


	//-----------------------------------------------------------------------------
	// Name: SaveMI(char* MI) 
	// Desc: Saves the ROM state
	//-----------------------------------------------------------------------------  	
	__declspec(dllexport) void SaveMI(char* MI) 
	{
		//Mapper 4
		//--------
		//  0     IRQ Counter
		//  1     IRQ Latch Counter
		//  2     (boolean) IRQ Counter Enabled
		//  3     Last value written to $8000
		MI[0] = IRQ_Counter;
		MI[1] = IRQ_Latch;
		MI[2] = IRQ_Enabled;
		MI[3] = Command;
	}

	//-----------------------------------------------------------------------------
	// Name: LoadMI(char *MI) 
	// Desc: Loads the Mapper ROM state
	//-----------------------------------------------------------------------------  
	__declspec(dllexport) void LoadMI(char* MI) 
	{
		IRQ_Counter = MI[0];
		IRQ_Latch   = MI[1];
		IRQ_Enabled = MI[2];
		Command     = MI[3];

		if ( (Command & 0x40) == 0 )
		{
			Bank0 = MP->GetPRG_Custom(0x08,8192);
			Bank2 = MP->GetPRG_Custom(0x0C,8192);
		}
		else
		{
			Bank2 = MP->GetPRG_Custom(0x08,8192);
			Bank0 = MP->GetPRG_Custom(0x0C,8192);
		}
	}

	//-----------------------------------------------------------------------------
	// Name: 
	// Desc: 
	//-----------------------------------------------------------------------------  
	__declspec(dllexport) NESTENMAPPER *LoadMapper(int VersionNeeded) 
	{

		// Check interface Version
		if (VersionNeeded > CurrentMapperInterface)
		{
			MessageBox(0, "DLL is not compatible with the current mapper interface!", "Mapper4", MB_OK);
			return NULL;
		}
		else
			return &mapper;
	}

	//-----------------------------------------------------------------------------
	// Name: 
	// Desc: 
	//-----------------------------------------------------------------------------  
	__declspec(dllexport) void UnloadMapper(void) 
	{
		// Nothing 4 now.
	}


	//-----------------------------------------------------------------------------
	// Name: InitMapper( TMapperParam *MapperParam )
	// Desc: Initialization of the mapper
	//-----------------------------------------------------------------------------  
	__declspec(dllexport) void InitMapper( TMapperParam *MapperParam )
	{
		// Copy the passed parameters to get access to the functions
		MP = MapperParam;

		MP->SetWriteHandler(0x08, CMD1Write);
		MP->SetWriteHandler(0x0A, CMD2Write);
		MP->SetWriteHandler(0x0C, CMD3Write);
		MP->SetWriteHandler(0x0E, CMD4Write);

		MP->SetPRG_89AB(0);  // Set first bank
		MP->SetPRG_CDEF(-1); // Set last bank

		Bank0 = MP->GetPRG_Custom(0x08,8192);
		Bank2 = MP->GetPRG_Custom(0x0C,8192);


		// Setup CHR ROM/RAM
		if (MP->HasCHR_ROM != 0)	MP->SetCHR_0000_1FFF(0);
		else						MP->SetCHR_RAM(0);

		// Mirroring
		if ((MP->Flag1 & 1) != 0)	MP->Mirror_V();
		else						MP->Mirror_H();
	}


	//-----------------------------------------------------------------------------
	// Name: HBlank(int ScanLine, int Byte2001)
	// Desc: called every hblank ( controls IRQ in mapper 4 )
	//-----------------------------------------------------------------------------  
	__declspec(dllexport) int HBlank(int ScanLine, int Byte2001)
	{
		
		if (ScanLine == 0)
		{
			IRQ_Counter = IRQ_Latch;
			return 0;
		}
	 
		if (ScanLine > 239) return 0; // Do nothing

		if ( (IRQ_Enabled != 0) && ((Byte2001 & 0x18) != 0) )
		{
			IRQ_Counter--;
			if (IRQ_Counter == 0x00)
			{				
				IRQ_Counter = IRQ_Latch;
				return 1;	//IRQ
			}
		}
		// if (Scanline == 240) IRQ_Counter = IRQ_Latch;
		return 0;
	}
}